/*
 * Decompiled with CFR 0.152.
 */
package ch.sparkpudding.coreengine;

import ch.sparkpudding.coreengine.Camera;
import ch.sparkpudding.coreengine.Input;
import ch.sparkpudding.coreengine.Lel;
import ch.sparkpudding.coreengine.ResourceLocator;
import ch.sparkpudding.coreengine.Scheduler;
import ch.sparkpudding.coreengine.api.Sound;
import ch.sparkpudding.coreengine.ecs.component.Component;
import ch.sparkpudding.coreengine.ecs.entity.Entity;
import ch.sparkpudding.coreengine.ecs.entity.Scene;
import ch.sparkpudding.coreengine.ecs.system.RenderSystem;
import ch.sparkpudding.coreengine.ecs.system.UpdateSystem;
import ch.sparkpudding.coreengine.filereader.LelReader;
import ch.sparkpudding.coreengine.filereader.XMLParser;
import ch.sparkpudding.coreengine.utils.Collision;
import ch.sparkpudding.coreengine.utils.Drawing;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.Toolkit;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Semaphore;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

public class CoreEngine
extends JPanel {
    private double msPerUpdate;
    private boolean exit;
    private Input input;
    private ResourceLocator resourceLocator;
    private LelReader lelFile;
    private Map<String, Scene> scenes;
    private Scene currentScene;
    private List<UpdateSystem> systems;
    private List<UpdateSystem> editingSystems;
    private RenderSystem renderSystem;
    private RenderSystem editingRenderSystem;
    private boolean pause;
    private boolean editingPause;
    private Dimension renderSize;
    private Color blackBarColor;
    private Semaphore renderLock;
    private int fpsCount;
    private int fps;
    private int editingTick;
    private Scheduler scheduler;
    private Exception gameError;

    public CoreEngine(String gameFolder) throws Exception {
        this.init(gameFolder);
        this.startGame();
        this.setCurrentScene(this.scenes.get("main"), false);
    }

    public CoreEngine(String gameFolder, String editingToolsFolder) throws Exception {
        this.init(gameFolder);
        this.lelFile.loadEditingTools(editingToolsFolder);
        this.populateEditingComponentTemplates();
        this.editingSystems = new ArrayList<UpdateSystem>();
        this.editingRenderSystem = this.loadSystemsFromFiles(this.editingSystems, this.lelFile.getEditingSystems());
        this.startGame();
        try {
            this.setCurrentScene(this.scenes.get("main"), false);
        }
        catch (Exception e) {
            this.notifyErrorAndClose("Could not find main scene.");
        }
    }

    private void init(String gameFolder) throws Exception {
        Lel.coreEngine = this;
        this.input = new Input(this);
        this.fps = 0;
        this.fpsCount = 0;
        this.msPerUpdate = 16.0;
        this.exit = false;
        this.editingTick = 0;
        this.pause = false;
        this.editingPause = false;
        this.scheduler = new Scheduler();
        this.renderSize = new Dimension(1280, 720);
        this.blackBarColor = Color.BLACK;
        this.renderLock = new Semaphore(0);
        this.lelFile = new LelReader(gameFolder);
        this.resourceLocator = new ResourceLocator(this.lelFile);
        this.populateComponentTemplates();
        this.populateEntityTemplates();
        this.populateScenes();
        this.systems = new ArrayList<UpdateSystem>();
        this.renderSystem = this.loadSystemsFromFiles(this.systems, this.lelFile.getSystems());
        if (this.renderSystem == null) {
            this.setEditingPause(true);
            this.notifyErrorAndClose("No render system found. Make sure to create at least one system named render.lua");
        }
    }

    private RenderSystem loadSystemsFromFiles(List<UpdateSystem> systemContainer, Collection<File> systemList) {
        RenderSystem renderSystem = null;
        systemContainer.clear();
        for (File systemFile : systemList) {
            if (systemFile.getName().equals("render.lua")) {
                renderSystem = new RenderSystem(systemFile);
                continue;
            }
            systemContainer.add(new UpdateSystem(systemFile));
        }
        return renderSystem;
    }

    private void populateScenes() {
        this.scenes = new HashMap<String, Scene>();
        for (File xmlFile : this.lelFile.getScenesXML()) {
            try {
                Scene scene = new Scene(XMLParser.parse(xmlFile));
                this.addScene(scene.getName(), scene);
            }
            catch (Exception e) {
                this.notifyErrorAndClose("The following error occured while parsing " + xmlFile + "\n\n" + e.getMessage());
            }
        }
        if (this.scenes.size() == 0) {
            this.notifyErrorAndClose("No scenes found. Please create at least one scene named main.xml");
        }
    }

    private void populateEntityTemplates() {
        for (File xmlFile : this.lelFile.getEntityTemplatesXML()) {
            try {
                Entity e = new Entity(XMLParser.parse(xmlFile));
                Entity.addTemplate(e);
            }
            catch (Exception e) {
                this.notifyErrorAndClose("The following error occured while parsing " + xmlFile + "\n\n" + e.getMessage());
            }
        }
        if (Entity.getTemplates().size() == 0) {
            this.notifyErrorAndClose("No entity templates found. Please create at least one entity template.");
        }
    }

    private void populateComponentTemplates() {
        for (File xmlFile : this.lelFile.getComponentsXML()) {
            try {
                Component c = new Component(XMLParser.parse(xmlFile));
                Component.addTemplate(c);
            }
            catch (Exception e) {
                this.notifyErrorAndClose("The following error occured while parsing " + xmlFile + "\n\n" + e.getMessage());
            }
        }
        if (Component.getTemplates().size() == 0) {
            this.notifyErrorAndClose("No component templates found. Please create at least one template.");
        }
    }

    private void populateEditingComponentTemplates() {
        for (File xmlFile : this.lelFile.getEditingComponentsXML()) {
            try {
                Component c = new Component(XMLParser.parse(xmlFile));
                Component.addTemplate(c);
            }
            catch (Exception e) {
                this.notifyErrorAndClose("The following error occured while parsing " + xmlFile + "\n\n" + e.getMessage());
            }
        }
    }

    private void startGame() throws InterruptedException {
        new Thread(() -> {
            try {
                this.runGameLoop();
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
        }).start();
    }

    private void runGameLoop() throws InterruptedException {
        double previous = System.currentTimeMillis();
        double lag = 0.0;
        boolean needRender = true;
        double lastFpsTime = System.currentTimeMillis();
        while (!this.exit) {
            this.scheduler.trigger(Scheduler.Trigger.GAME_LOOP_START);
            double current = System.currentTimeMillis();
            double elapsed = current - previous;
            previous = current;
            lag += elapsed;
            needRender = false;
            while (lag >= this.msPerUpdate) {
                this.input.update();
                needRender = true;
                if (this.editingSystems == null) {
                    this.handleLuaErrors();
                }
                if (this.gameError == null) {
                    this.update();
                }
                lag -= this.msPerUpdate;
            }
            if (needRender) {
                this.render();
                this.renderLock.acquire();
            }
            if (!((double)System.currentTimeMillis() - lastFpsTime >= 1000.0)) continue;
            lastFpsTime = System.currentTimeMillis();
            this.fpsCount = this.fps;
            this.fps = 0;
        }
    }

    private void handleLuaErrors() {
        if (this.gameError != null) {
            if (this.input.isKeyDown(32)) {
                this.setEditingPause(false);
                this.clearError();
            } else if (this.input.isKeyDown(10)) {
                this.reloadSystemsFromDisk();
                this.clearError();
                this.setEditingPause(false);
            }
            this.input.resetAllKeys();
        }
    }

    public void clearError() {
        this.gameError = null;
    }

    public void reloadSystemsFromDisk() {
        if (this.editingSystems != null) {
            this.editingRenderSystem = this.loadSystemsFromFiles(this.editingSystems, this.lelFile.getEditingSystems());
        }
        this.renderSystem = this.loadSystemsFromFiles(this.systems, this.lelFile.getSystems());
        this.setCurrentScene(this.getCurrentScene(), false);
    }

    private void update() {
        this.scheduler.trigger(Scheduler.Trigger.BEFORE_UPDATE);
        this.currentScene.getCamera().update();
        if (this.editingPause && this.editingSystems != null) {
            ++this.editingTick;
            for (UpdateSystem system : this.editingSystems) {
                system.update();
            }
        } else {
            this.currentScene.incrementTick();
            for (UpdateSystem system : this.systems) {
                system.update();
            }
        }
    }

    private void render() {
        this.repaint();
    }

    public boolean isEditingPause() {
        return this.editingPause;
    }

    public void setEditingPause(boolean pause) {
        this.scheduler.trigger(Scheduler.Trigger.EDITING_STATE_CHANGED, pause);
        this.editingPause = pause;
    }

    public void setPause(boolean isPause) {
        this.pause = isPause;
    }

    public void setEditingPause(boolean pause, boolean noTrigger) {
        if (!noTrigger) {
            this.scheduler.trigger(Scheduler.Trigger.EDITING_STATE_CHANGED, pause);
        }
        this.editingPause = pause;
        if (this.editingPause) {
            Sound.getInstance().pauseMusic();
        } else {
            Sound.getInstance().resumeMusic();
        }
    }

    public void togglePause() {
        this.pause = !this.pause;
    }

    public Map<String, Scene> getScenes() {
        return this.scenes;
    }

    public void addScene(String name, Scene s) {
        this.scenes.put(name, s);
        this.scheduler.trigger(Scheduler.Trigger.SCENE_LIST_CHANGED);
    }

    public void deleteScene(final String name) {
        if (this.currentScene.getName().equals(name)) {
            this.scheduler.schedule(Scheduler.Trigger.GAME_LOOP_START, new Runnable(){

                @Override
                public void run() {
                    CoreEngine.this.setCurrentScene((Scene)CoreEngine.this.scenes.get("main"), false);
                    CoreEngine.this.scenes.remove(name);
                    CoreEngine.this.scheduler.trigger(Scheduler.Trigger.SCENE_LIST_CHANGED);
                }
            });
        } else {
            this.scenes.remove(name);
            this.scheduler.trigger(Scheduler.Trigger.SCENE_LIST_CHANGED);
        }
    }

    public void setScene(String name, boolean reset, boolean useDirectlyDefaultEntities) {
        if (reset) {
            this.scenes.get(name).reset(useDirectlyDefaultEntities);
        }
        this.setCurrentScene(this.scenes.get(name), useDirectlyDefaultEntities);
    }

    public void resetCurrentScene(boolean useDirectlyDefaultEntities) {
        this.editingTick = 0;
        Sound.getInstance().stopMusic();
        this.currentScene.reset(useDirectlyDefaultEntities);
        this.setCurrentScene(this.currentScene, useDirectlyDefaultEntities);
    }

    public boolean renameScene(String oldName, String newName) {
        Scene scene = this.scenes.get(oldName);
        if (scene != null) {
            this.scenes.remove(oldName);
            scene.setName(newName);
            this.scenes.put(newName, scene);
            this.scheduler.trigger(Scheduler.Trigger.SCENE_LIST_CHANGED);
            return true;
        }
        return false;
    }

    public Scheduler getScheduler() {
        return this.scheduler;
    }

    public Scene getCurrentScene() {
        return this.currentScene;
    }

    public void setCurrentScene(Scene newScene, boolean useDirectlyDefaultEntities) {
        this.currentScene = newScene;
        List<Entity> entities = useDirectlyDefaultEntities ? newScene.getDefaultEntities() : newScene.getEntities();
        for (UpdateSystem system : this.systems) {
            system.setEntities(entities);
        }
        if (this.renderSystem != null) {
            this.renderSystem.setEntities(entities);
        }
        if (this.editingSystems != null) {
            for (UpdateSystem system : this.editingSystems) {
                system.setEntities(entities);
            }
        }
        if (this.editingRenderSystem != null) {
            this.editingRenderSystem.setEntities(entities);
        }
        this.scheduler.trigger(Scheduler.Trigger.SCENE_CHANGED, newScene);
    }

    private Point getGameTranslation() {
        double scaleRatio = this.getScaleRatio();
        int realGameWidth = (int)(scaleRatio * this.renderSize.getWidth());
        int realGameHeight = (int)(scaleRatio * this.renderSize.getHeight());
        int translateX = this.getWidth() / 2 - realGameWidth / 2;
        int translateY = this.getHeight() / 2 - realGameHeight / 2;
        return new Point(translateX, translateY);
    }

    private double getScaleRatio() {
        double scaleRatio = 1.0;
        double heightScaleRatio = (double)this.getHeight() / this.renderSize.getHeight();
        double widthScaleRatio = (double)this.getWidth() / this.renderSize.getWidth();
        scaleRatio = widthScaleRatio > heightScaleRatio ? heightScaleRatio : widthScaleRatio;
        return scaleRatio;
    }

    @Override
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        Graphics2D g2d = (Graphics2D)g;
        AffineTransform defaultTransformationState = g2d.getTransform();
        g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        double scaleRatio = this.getScaleRatio();
        Point translation = this.getGameTranslation();
        g2d.translate(translation.getX(), translation.getY());
        g2d.scale(scaleRatio, scaleRatio);
        AffineTransform gameTransformationState = g2d.getTransform();
        ++this.fps;
        if (this.renderSystem != null) {
            this.renderSystem.render(g2d);
        }
        if (this.editingPause && this.editingRenderSystem != null) {
            g2d.setTransform(gameTransformationState);
            this.editingRenderSystem.render(g2d);
        }
        g2d.setTransform(defaultTransformationState);
        this.drawBlackBars(g2d, scaleRatio, translation);
        this.renderLuaError(g2d);
        Toolkit.getDefaultToolkit().sync();
        g.dispose();
        this.renderLock.release();
    }

    private void renderLuaError(Graphics2D g2d) {
        if (this.gameError != null) {
            int x = 20;
            int y = 40;
            int maxWidth = this.getWidth() - x;
            int smallFontSize = 15;
            int bigFontSize = 30;
            g2d.setColor(new Color(0, 0, 0, 150));
            g2d.fillRect(0, 0, this.getWidth(), this.getHeight());
            g2d.setColor(new Color(0, 0, 0, 255));
            g2d.fillRect(0, 0, this.getWidth(), 170);
            g2d.setColor(Color.WHITE);
            g2d.setFont(new Font("Dialog", 1, bigFontSize));
            y = Drawing.drawWrappedString("Something went wrong", x, y, maxWidth, g2d);
            g2d.setColor(Color.RED);
            g2d.setFont(new Font("SansSerif", 0, smallFontSize));
            y = Drawing.drawWrappedString(this.gameError.getMessage(), x, y, maxWidth, g2d);
            if (this.editingSystems == null) {
                y += smallFontSize;
                g2d.setColor(Color.LIGHT_GRAY);
                y = Drawing.drawWrappedString("Press [SPACE] to ignore this error and attempt to continue.", x, y, maxWidth, g2d);
                Drawing.drawWrappedString("Press [ENTER] to reload systems from disk.", x, y, maxWidth, g2d);
            }
        }
    }

    private void drawBlackBars(Graphics2D g2d, double scaleRatio, Point translation) {
        g2d.setColor(this.blackBarColor);
        double heightScaleRatio = (double)this.getHeight() / this.renderSize.getHeight();
        double widthScaleRatio = (double)this.getWidth() / this.renderSize.getWidth();
        int realGameWidth = (int)(scaleRatio * this.renderSize.getWidth());
        int realGameHeight = (int)(scaleRatio * this.renderSize.getHeight());
        int translateX = (int)translation.getX();
        int translateY = (int)translation.getY();
        if (widthScaleRatio > heightScaleRatio) {
            g2d.fillRect(0, 0, translateX, this.getHeight());
            g2d.fillRect(translateX + realGameWidth, 0, translateX + 1, this.getHeight() + 1);
        } else {
            g2d.fillRect(0, 0, this.getWidth(), translateY);
            g2d.fillRect(0, translateY + realGameHeight, this.getWidth() + 1, translateY + 1);
        }
    }

    public Camera getCamera() {
        return this.currentScene.getCamera();
    }

    public Input getInput() {
        return this.input;
    }

    public ResourceLocator getResourceLocator() {
        return this.resourceLocator;
    }

    public RenderSystem getRenderSystems() {
        return this.renderSystem;
    }

    public List<UpdateSystem> getSystems() {
        return this.systems;
    }

    public void addEntity(Entity e) {
        this.renderSystem.tryAdd(e);
        for (UpdateSystem system : this.systems) {
            system.tryAdd(e);
        }
        if (this.editingRenderSystem != null) {
            this.editingRenderSystem.tryAdd(e);
        }
        if (this.editingSystems != null) {
            for (UpdateSystem system : this.editingSystems) {
                system.tryAdd(e);
            }
        }
        this.getCurrentScene().addEntity(e);
    }

    public void notifyZIndexChange() {
        this.renderSystem.sortEntities();
        for (UpdateSystem system : this.systems) {
            system.sortEntities();
        }
    }

    public int getFPS() {
        return this.fpsCount;
    }

    public void deleteEntity(Entity entity) {
        for (UpdateSystem system : this.systems) {
            system.tryRemove(entity);
        }
        this.renderSystem.tryRemove(entity);
        this.currentScene.removeEntity(entity);
        if (this.editingSystems != null) {
            for (UpdateSystem system : this.editingSystems) {
                system.tryRemove(entity);
            }
        }
        if (this.editingRenderSystem != null) {
            this.editingRenderSystem.tryRemove(entity);
        }
    }

    public void deleteDefaultEntity(Entity entity) {
        this.currentScene.removeDefaultEntity(entity);
    }

    public void deleteComponent(Entity entity, String componentName) {
        if (entity.removeComponent(componentName)) {
            for (UpdateSystem system : this.systems) {
                system.notifyRemovedComponent(entity, componentName);
            }
            this.renderSystem.notifyRemovedComponent(entity, componentName);
            if (this.editingSystems != null) {
                for (UpdateSystem system : this.editingSystems) {
                    system.notifyRemovedComponent(entity, componentName);
                }
            }
            if (this.editingRenderSystem != null) {
                this.editingRenderSystem.notifyRemovedComponent(entity, componentName);
            }
        }
    }

    public void notifySystemsOfNewComponent(Entity entity, Component component) {
        String componentName = component.getName();
        for (UpdateSystem system : this.systems) {
            system.notifyNewComponent(entity, componentName);
        }
        this.renderSystem.notifyNewComponent(entity, componentName);
        if (this.editingSystems != null) {
            for (UpdateSystem system : this.editingSystems) {
                system.notifyNewComponent(entity, componentName);
            }
        }
        if (this.editingRenderSystem != null) {
            this.editingRenderSystem.notifyNewComponent(entity, componentName);
        }
    }

    public double getGameHeight() {
        return this.renderSize.getHeight();
    }

    public double getGameWidth() {
        return this.renderSize.getWidth();
    }

    public Point2D panelPositionToGame(Point2D p) {
        double x = p.getX();
        double y = p.getY();
        double scaleRatio = this.getScaleRatio();
        Point translation = this.getGameTranslation();
        x -= translation.getX();
        y -= translation.getY();
        return new Point2D.Double(x /= scaleRatio, y /= scaleRatio);
    }

    public Point2D panelPositionToWorld(Point2D p) {
        Point2D gamePosition = this.panelPositionToGame(p);
        double x = gamePosition.getX();
        double y = gamePosition.getY();
        x += this.currentScene.getCamera().getPosition().getX();
        y += this.currentScene.getCamera().getPosition().getY();
        return new Point2D.Double(x /= (double)this.currentScene.getCamera().getScaling(), y /= (double)this.currentScene.getCamera().getScaling());
    }

    public Point2D panelVectorToGame(Point2D v) {
        double dx = v.getX();
        double dy = v.getY();
        double scaleRatio = this.getScaleRatio();
        return new Point2D.Double(dx /= scaleRatio, dy /= scaleRatio);
    }

    public Point2D panelVectorToWorld(Point2D v) {
        double dx = v.getX();
        double dy = v.getY();
        return new Point2D.Double(dx /= (double)this.currentScene.getCamera().getScaling(), dy /= (double)this.currentScene.getCamera().getScaling());
    }

    public List<Entity> getEntitiesAt(Point2D p) {
        ArrayList<Entity> entities = new ArrayList<Entity>();
        ArrayList<String> requiredComponents = new ArrayList<String>();
        requiredComponents.add("position");
        requiredComponents.add("size");
        for (Entity entity : this.currentScene.getEntities()) {
            Map<String, Component> components;
            if (!entity.hasComponents(requiredComponents) || (components = entity.getComponents()).get("position") == null || components.get("size") == null || !Collision.intersectRect(p.getX(), p.getY(), components.get("position").getField("x").getDouble(), components.get("position").getField("y").getDouble(), components.get("size").getField("width").getDouble(), components.get("size").getField("height").getDouble())) continue;
            entities.add(entity);
        }
        return entities;
    }

    public void notifyGameError(final Exception exception) {
        if (this.gameError == null) {
            this.scheduler.schedule(Scheduler.Trigger.GAME_LOOP_START, new Runnable(){

                @Override
                public void run() {
                    CoreEngine.this.gameError = exception;
                    CoreEngine.this.setEditingPause(true);
                }
            });
        }
    }

    public void notifyErrorAndClose(String string) {
        JOptionPane.showMessageDialog(this, string, "A fatal error occured", 0);
        System.exit(1);
    }

    public void setBlackBarsColor(Color color) {
        this.blackBarColor = color;
    }

    public boolean isInError() {
        return this.gameError != null;
    }

    public int getEditingTick() {
        return this.editingTick;
    }

    public String getGameFolder() {
        return this.lelFile.getDirectory();
    }
}

